<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/


/**
 * Form element for GPG public key
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 */

class LSformElement_gpg_pub_key extends LSformElement {

  var $template = 'LSformElement_gpg_pub_key.tpl';
  var $fieldTemplate = 'LSformElement_gpg_pub_key_field.tpl';


  /**
   * Parse one value
   *
   * @param string $value The value to parse
   * @param boolean $details Enable/disable details return (optional, default: true)
   *
   * @return array|string Parsed value as array is $details is enabled, the raw value otherwise
   */
  public function parseValue($value, $details=true) {
    if (!$details)
      return $value;
    if (class_exists('gnupg')) {
      // The home_dir parameter passed to gnupg_init() is not correctly handled in PHP 7.3, also set
      // the GNUPGHOME environment variable.
      putenv('GNUPGHOME='.LS_TMP_DIR_PATH);
      $gpg = new gnupg(["home_dir" => LS_TMP_DIR_PATH]);
      // Don't warn about (GNUPG_ERROR_SILENT is currently the default but ensure it)
      $gpg -> seterrormode(GNUPG_ERROR_SILENT);
      $info = $gpg -> import($value);
    }
    else {
      LSerror :: addErrorCode('LSformElement_gpg_pub_key_01');
      $info = false;
    }
    if (
      is_array($info)
      // @phpstan-ignore-next-line
      && ($info['imported'] + $info['unchanged']) != 1
    )
      $info = false;
    return array(
      'fingerprint' => is_array($info)?$info["fingerprint"]:null,
      'value' => $value
    );
  }

 /**
  * Retourne les infos d'affichage de l'élément
  *
  * Cette méthode retourne les informations d'affichage de l'élement
  *
  * @return array
  */
  public function getDisplay(){
    LStemplate :: addCssFile('LSformElement_gpg_pub_key.css');
    $return = $this -> getLabelInfos();
    $params = array();
    if (!$this -> isFreeze()) {
      $params['values_txt'] = $this -> values;
    }
    else {
      LStemplate :: addJSscript('LSformElement_gpg_pub_key.js');
      LStemplate :: addHelpInfo(
        'LSformElement_gpg_pub_key',
        array(
          'display' => _("Display the full key.")
        )
      );

      $values_txt = array();
      foreach ($this -> values as $value) {
        $parsedValue = $this -> parseValue($value);
        $values_txt[] = $parsedValue;
      }
      $params['values_txt'] = $values_txt;
      $params['invalidValueTxt'] = _('Invalid value');
    }
    $return['html'] = $this -> fetchTemplate(NULL, $params);
    return $return;
  }

}

/*
 * Error Codes
 */
LSerror :: defineError('LSformElement_gpg_pub_key_01',
___("LSformElement_gpg_pub_key: PHP GnuPG extension is missing, can't parse value.")
);
