# Les commandes *CLI* personnalisées

Les [LSaddons](../../conf/index.md#configuration-des-lsaddons) peuvent fournir des commandes *CLI*
personnalisées qui seront accessibles via la commande `ldapsaisie` fournie avec l'application. Cela
peut, par exemple, vous permettre de rendre accessible en ligne de commandes une procédure
implémentée dans le code d'LdapSaisie et vous permettre de mettre en place une tâche planifiée
exécutant cette procédure régulièrement.

Pour mettre en place une telle commande *CLI* personnalisée, il est nécessaire de :

- Déclarer cette vue dans la fonction `LSaddon_[addon]_support` de l'addon à l'aide de la méthode
  `LScli :: add_command()` ;

- Déclarer la fonction implémentant cette commande *CLI* personnalisée. Cette fonction acceptera,
  en tant qu'unique paramètre, un tableau des arguments reçus lors de l'exécution de la commande et
  retournera `True` ou `False` en cas de succès/d'erreur d'exécution de la commande. Cette valeur de
  retour influencera le code retourné par la commande : `0` en cas de succès, `1` en cas d'erreur.

- Bien que cela ne soit pas obligatoire, il sera également possible de déclarer une fonction
  permettant l'autocomplétion des arguments acceptés par la commande.

  Cette méthode recevra en paramètre :

  - `$command_args`

    Un tableau des arguments déjà reçus par la commande.

  - `$comp_word_num`

    Un entier indiquant le rang de l'argument que l'autocomplétion tente de compléter. Il peut
    s'agir du rang d'un paramètre déjà fourni et présent dans le tableau `$command_args` ou bien
    d'un rang supérieur aux nombres d'arguments déjà fournis à la commande et dans ce cas il s'agira
    d'autocompléter tous potentiels autre argument que pourrait accepter cette commande.

  - `$comp_word`

    Une chaîne de caractères correspondant à ce qu'a déjà saisi l'utilisateur de l'argument que l'on
    tente d'autocompléter. Cette chaîne de caractères peut être vide ou non, en fonction de s'il
    s'agit d'un nouvel argument à autocompléter ou non.

  - `$opts`

    Un tableau des potentiels arguments globaux acceptés par *LScli* dans le contexte actuel (par
    exemple, `-d` ou `--debug` pour l'activation du mode debug). La réponse de cette fonction devra
    inclure ces potentiels arguments si le contexte d'autocomplétion si prête (nouvel argument par
    exemple).

  Pour finir, cette fonction devra retourner un tableau des potentielles valeurs que pourrait
  prendre l'argument autocomplété. Si une unique proposition est faite à l'utilisateur, celle-ci
  sera automatiquement proposée à l'utilisateur et à défaut, la liste des valeurs possibles lui
  seront affichées.

  !!! note

      Pour vous aider dans l'écrire d'une telle méthode d'autocomplétion, des méthodes statiques
      sont fournies par la classe `LScli` pour les autocomplétions les plus courantes :

      - `LScli :: autocomplete_class_name()`

        Autocomplétion du nom d'une classe PHP.

      - `LScli :: autocomplete_addon_name()`

        Autocomplétion du nom d'un [LSaddon](../../conf/index.md#configuration-des-lsaddons).

      - `LScli :: autocomplete_int()`

        Autocomplétion d'un nombre entier.

      - `LScli :: autocomplete_LSobject_types()`

        Autocomplétion du nom d'un type d'[LSobject](../../conf/index.md#configuration-lsobject).

      - `LScli :: autocomplete_LSobject_dn()`

        Autocomplétion du DN d'un type précis d'[LSobject](../../conf/index.md#configuration-lsobject) de
        l'annuaire.

      Par ailleurs, la méthode `LScli :: autocomplete_opts()` vous facilitera la construction de la
      liste des valeurs d'autocomplétion de l'argument courant en fonction de ce qui a déjà été
      saisi par l'utilisateur (paramètre `$comp_word`). Cette méthode s'occupera en l'occurrence de
      filtrer parmi toutes les valeurs contextuelles possibles, celles qui correspondent au préfixe
      fourni par l'utilisateur.

Pour implémenter une telle commande *CLI* personnalisée, vous pouvez vous inspirer de l'exemple
fourni ci-dessous ou encore des commandes *CLI* fournies par les autres
[LSaddons](../../conf/index.md#configuration-des-lsaddons) ou classes PHP de l'application.

**Structure du fichier includes/addons/LSaddons.[addon name].php :**

```
<?php
function LSaddon_myaddon_support() {

  $retval=true;

  // Some check

  if ($retval) {
    if (php_sapi_name() == 'cli') {
        LScli :: add_command(
          'my_custom_cli_cmd',                    // The CLI command name (required)
          'cli_my_custom_cli_cmd',                // The CLI command handler (must be callable, required)
          'My custom CLI command',                // A short description of what this command does (required)
          '[arg1] [arg2] [...]',                  // A short list of commands available arguments show in usage message
                                                  // (optional, default: false)
          'This command permit to ...',           // A long description of what this command does (optional, default:
                                                  // false)
          true,                                   // Permit to define if this command need connection to LDAP server
                                                  // (optional, default: true)
          'cli_my_custom_cli_cmd_autocompleter',  // Callable of the CLI command arguments autocompleter (optional,
                                                  // default: null)
          true                                    // Allow override if a command already exists with the same name
                                                  // (optional, default: null)
        );
    }
  }

  return $retval;
}

[...]

// Defined CLI commands functions only on CLI context
if (php_sapi_name() != 'cli')
    return true;  // Always return true to avoid some warning in log

/**
 * My addon CLI command my_custom_cli_cmd handler function
 *
 * Description of this CLI command.
 *
 * @param array $command_args Command arguments
 *   - Positional arguments :
 *     - LSobject
 *     - dn
 *   - Optional arguments :
 *     - -f|--force : Force mode
 *
 * @author My Name <my.email@example.com>
 *
 * @return boolean True on success, false otherwise
 **/
function cli_my_custom_cli_cmd($command_args) {
    $objType = null;
    $dn = null;
    $force_mode = false;
    foreach ($command_args as $arg) {
      if ($arg == '-f' || $arg == '--force')
        $force_mode = true;
      elseif (is_null($objType)) {
        $objType = $arg;
      }
      elseif (is_null($dn)) {
        $dn = $arg;
      }
      else
        LScli :: usage("Invalid $arg parameter.");
    }

    if (is_null($objType) || is_null($dn))
      LScli :: usage('You must provide LSobject type and DN.');

    if (!LSsession :: loadLSobject($objType))
      return false;

    $obj = new $objType();
    if (!$obj->loadData($dn)) {
      self :: log_fatal("Fail to load object $dn data from LDAP");
      return false;
    }

    // Do some stuff on loaded object
    [...]

    return true;
}

/**
 * Args autocompleter for CLI my_custom_cli_cmd command
 *
 * @param array<string> $command_args List of already typed words of the command
 * @param int $comp_word_num The command word number to autocomplete
 * @param string $comp_word The command word to autocomplete
 * @param array<string> $opts List of global available options
 *
 * @return array<string> List of available options for the word to autocomplete
 **/
public static function cli_my_custom_cli_cmd_autocompleter($command_args, $comp_word_num, $comp_word, $opts) {
  $opts = array_merge($opts, array ('-f', '--force'));

  // Handle positional args
  $objType = null;
  $objType_arg_num = null;
  $dn = null;
  $dn_arg_num = null;
  for ($i=0; $i < count($command_args); $i++) {
    if (!in_array($command_args[$i], $opts)) {
      // If object type not defined
      if (is_null($objType)) {
        // Defined it
        $objType = $command_args[$i];
        LScli :: unquote_word($objType);
        $objType_arg_num = $i;

        // Check object type exists
        $objTypes = LScli :: autocomplete_LSobject_types($objType);

        // Load it if exist and not trying to complete it
        if (in_array($objType, $objTypes) && $i != $comp_word_num) {
          LSsession :: loadLSobject($objType, false);
        }
      }
      elseif (is_null($dn)) {
        $dn = $command_args[$i];
        LScli :: unquote_word($dn);
        $dn_arg_num = $i;
      }
    }
  }

  // If objType not already choiced (or currently autocomplete), add LSobject types to available options
  if (!$objType || $objType_arg_num == $comp_word_num)
    $opts = array_merge($opts, LScli :: autocomplete_LSobject_types($comp_word));

  // If dn not alreay choiced (or currently autocomplete), try autocomplete it
  elseif (!$dn || $dn_arg_num == $comp_word_num)
    $opts = array_merge($opts, LScli :: autocomplete_LSobject_dn($objType, $comp_word));

  return LScli :: autocomplete_opts($opts, $comp_word);
}
```
